# Copyright (c) 2022 MetPy Developers.
# Distributed under the terms of the BSD 3-Clause License.
# SPDX-License-Identifier: BSD-3-Clause
"""Collection of pre-defined areas for plotting with associated projections."""

from collections import namedtuple

Area = namedtuple('Area', ['name', 'description', 'bounds', 'projection'])

try:
    import cartopy.crs as ccrs

    area_data = [
        Area('us', 'UNITED STATES', (-119.0, -56.0, 19.0, 47.0),
             ccrs.NorthPolarStereo(central_longitude=-100)),
        Area('awips', 'AWIPS US', (-127.0, -59.0, 20.0, 50.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-100)),
        Area('uslcc', 'UNITED STATES LCC', (-124.0, -62.0, 20.0, 51.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-100)),
        Area('whlf', 'WESTERN HALF', (-123.8, -85.9, 22.9, 50.2),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-105)),
        Area('chlf', 'CENTRAL HALF', (-111.0, -79.0, 27.5, 50.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-100)),
        Area('centus', 'CENTRAL US', (-105.4, -77.0, 24.7, 47.6),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-100)),
        Area('ehlf', 'EASTERN HALF', (-96.2, -62.7, 22.0, 49.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-90)),
        Area('mehlf', 'EASTERN HALF', (-89.9, -66.6, 23.8, 49.1),
             ccrs.PlateCarree()),
        Area('west', 'WESTERN US', (-125.0, -90.0, 25.0, 55.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-105)),
        Area('cent', 'CENTRAL US', (-107.4, -75.3, 24.3, 49.7),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-96)),
        Area('east', 'EASTERN US', (-100.55, -65.42, 24.57, 47.2),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-90)),
        Area('nwus', 'NW SECTOR', (-126.0, -102.0, 38.25, 50.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-114)),
        Area('swus', 'SW SECTOR', (-126.0, -100.0, 28.25, 40.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-110)),
        Area('ncus', 'NC SECTOR', (-108.0, -84.0, 38.25, 50.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-98)),
        Area('scus', 'SC SECTOR', (-108.9, -84.0, 24.0, 40.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-100)),
        Area('neus', 'NE SECTOR', (-89.0, -64.0, 37.25, 47.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-78)),
        Area('seus', 'SE SECTOR', (-90.0, -66.0, 28.25, 40.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-87)),
        Area('bosfa', 'BOSTON FA AREA', (-87.5, -63.5, 34.5, 50.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-75)),
        Area('miafa', 'MIAMI FA AREA', (-88.0, -72.0, 23.0, 39.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-80)),
        Area('chifa', 'CHICAGO FA AREA', (-108.0, -75.0, 34.0, 50.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-93)),
        Area('dfwfa', 'DALLAS FA AREA', (-106.5, -80.5, 22.0, 40.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-99)),
        Area('slcfa', 'SALT LAKE FA AREA', (-126.0, -98.0, 29.5, 50.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-109)),
        Area('sfofa', 'SAN FRAN FA AREA', (-129.0, -111.0, 30.0, 50.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-118)),
        Area('ak', 'ALASKA HPC QPF', (-180.0, -106.0, 42.0, 73.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-143)),
        Area('al', 'ALABAMA', (-95.0, -79.0, 27.0, 38.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-87)),
        Area('ar', 'ARKANSAS', (-100.75, -84.75, 29.5, 40.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-93)),
        Area('ca', 'CALIFORNIA', (-127.75, -111.75, 31.5, 42.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-120)),
        Area('co', 'COLORADO', (-114.0, -98.0, 33.5, 44.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-106)),
        Area('ct', 'CONNECTICUT', (-81.25, -65.25, 36.0, 47.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-73)),
        Area('dc', 'WASHINGTON DC', (-85.0, -69.0, 33.35, 44.35),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-77)),
        Area('de', 'DELAWARE', (-83.75, -67.75, 33.25, 44.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-76)),
        Area('fl', 'FLORIDA', (-90.0, -74.0, 23.0, 34.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-82)),
        Area('ga', 'GEORGIA', (-92.0, -76.0, 27.5, 38.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-84)),
        Area('hi', 'HAWAII', (-161.5, -152.5, 17.0, 23.0),
             ccrs.PlateCarree()),
        Area('ia', 'IOWA', (-102.0, -86.0, 36.5, 47.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-94)),
        Area('id', 'IDAHO', (-123.0, -107.0, 39.25, 50.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-115)),
        Area('il', 'ILLINOIS', (-97.75, -81.75, 34.5, 45.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-90)),
        Area('in', 'INDIANA', (-94.5, -78.5, 34.5, 45.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-87)),
        Area('ks', 'KANSAS', (-106.5, -90.5, 33.25, 44.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-99)),
        Area('ky', 'KENTUCKY', (-93.0, -77.0, 31.75, 42.75),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-85)),
        Area('la', 'LOUISIANA', (-100.75, -84.75, 25.75, 36.75),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-93)),
        Area('ma', 'MASSACHUSETTS', (-80.25, -64.25, 36.75, 47.75),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-72)),
        Area('md', 'MARYLAND', (-85.25, -69.25, 33.75, 44.75),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-77)),
        Area('me', 'MAINE', (-77.75, -61.75, 39.5, 50.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-70)),
        Area('mi', 'MICHIGAN', (-93.0, -77.0, 37.75, 48.75),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-85)),
        Area('mn', 'MINNESOTA', (-102.0, -86.0, 40.5, 51.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-94)),
        Area('mo', 'MISSOURI', (-101.0, -85.0, 33.0, 44.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-93)),
        Area('ms', 'MISSISSIPPI', (-98.0, -82.0, 27.0, 38.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-90)),
        Area('mt', 'MONTANA', (-117.0, -101.0, 41.5, 52.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-109)),
        Area('nc', 'NORTH CAROLINA', (-87.25, -71.25, 30.0, 41.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-79)),
        Area('nd', 'NORTH DAKOTA', (-107.5, -91.5, 42.25, 53.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-100)),
        Area('ne', 'NEBRASKA', (-107.5, -91.5, 36.25, 47.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-100)),
        Area('nh', 'NEW HAMPSHIRE', (-79.5, -63.5, 38.25, 49.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-72)),
        Area('nj', 'NEW JERSEY', (-82.5, -66.5, 34.75, 45.75),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-75)),
        Area('nm', 'NEW MEXICO', (-114.25, -98.25, 29.0, 40.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-106)),
        Area('nv', 'NEVADA', (-125.0, -109.0, 34.0, 45.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-117)),
        Area('ny', 'NEW YORK', (-84.0, -68.0, 37.25, 48.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-76)),
        Area('oh', 'OHIO', (-91.0, -75.0, 34.5, 45.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-83)),
        Area('ok', 'OKLAHOMA', (-105.25, -89.25, 30.25, 41.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-97)),
        Area('or', 'OREGON', (-128.0, -112.0, 38.75, 49.75),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-120)),
        Area('pa', 'PENNSYLVANIA', (-86.0, -70.0, 35.5, 46.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-78)),
        Area('ri', 'RHODE ISLAND', (-79.75, -63.75, 36.0, 47.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-72)),
        Area('sc', 'SOUTH CAROLINA', (-89.0, -73.0, 28.5, 39.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-81)),
        Area('sd', 'SOUTH DAKOTA', (-107.5, -91.5, 39.0, 50.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-100)),
        Area('tn', 'TENNESSEE', (-95.0, -79.0, 30.0, 41.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-87)),
        Area('tx', 'TEXAS', (-107.0, -91.0, 25.4, 36.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-99)),
        Area('ut', 'UTAH', (-119.0, -103.0, 34.0, 45.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-111)),
        Area('va', 'VIRGINIA', (-86.5, -70.5, 32.25, 43.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-79)),
        Area('vt', 'VERMONT', (-80.75, -64.75, 38.25, 49.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-73)),
        Area('wi', 'WISCONSIN', (-98.0, -82.0, 38.5, 49.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-90)),
        Area('wv', 'WEST VIRGINIA', (-89.0, -73.0, 33.0, 44.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-81)),
        Area('wy', 'WYOMING', (-116.0, -100.0, 37.75, 48.75),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-108)),
        Area('az', 'ARIZONA', (-119.0, -103.0, 29.0, 40.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-111)),
        Area('wa', 'WASHINGTON', (-128.0, -112.0, 41.75, 52.75),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-120)),
        Area('cn', 'CANADA', (-120.4, -14.0, 37.9, 58.6),
             ccrs.NorthPolarStereo(central_longitude=-90)),
        Area('cnlcc', 'CANADA LCC', (-127, -54.0, 40, 70),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-100)),
        Area('ab', 'ALBERTA', (-119.6, -108.2, 48.6, 60.4),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-114)),
        Area('bc', 'BRITISH COLUMBIA', (-134.5, -109.0, 47.2, 60.7),
             ccrs.NorthPolarStereo(central_longitude=-122)),
        Area('mb', 'MANITOBA', (-102.4, -86.1, 48.3, 60.2),
             ccrs.NorthPolarStereo(central_longitude=-95)),
        Area('nb', 'NEW BRUNSWICK', (-75.7, -57.6, 42.7, 49.6),
             ccrs.NorthPolarStereo(central_longitude=-67)),
        Area('nf', 'NEWFOUNDLAND', (-68.0, -47.0, 45.0, 62.0),
             ccrs.NorthPolarStereo(central_longitude=-58)),
        Area('ns', 'NOVA SCOTIA', (-67.0, -59.0, 43.0, 47.5),
             ccrs.NorthPolarStereo(central_longitude=-63)),
        Area('nt', 'NW TERRITORIES', (-132.0, -100.0, 59.0, 70.5),
             ccrs.NorthPolarStereo(central_longitude=-112)),
        Area('nu', 'NUNAVUT', (-115.0, -65.0, 59.0, 80),
             ccrs.NorthPolarStereo(central_longitude=-95)),
        Area('on', 'ONTARIO', (-94.5, -68.2, 41.9, 55.0),
             ccrs.NorthPolarStereo(central_longitude=-90)),
        Area('pe', 'PRINCE EDWARD IS', (-64.6, -61.7, 45.8, 47.1),
             ccrs.NorthPolarStereo(central_longitude=-63)),
        Area('qb', 'QUEBEC', (-80.0, -49.2, 44.1, 60.9),
             ccrs.NorthPolarStereo(central_longitude=-74)),
        Area('sa', 'SASKATCHEWAN', (-111.2, -97.8, 48.5, 60.3),
             ccrs.NorthPolarStereo(central_longitude=-105)),
        Area('yt', 'YUKON TERRITORY', (-142.0, -117.0, 59.0, 70.5),
             ccrs.NorthPolarStereo(central_longitude=-130)),
        Area('tropatl', 'TROPICAL ATLANTIC', (-115.0, 10.0, 0.0, 40.0),
             ccrs.PlateCarree()),
        Area('subtrop', 'SUBTROPICAL ATL', (-90.0, -20.0, 20.0, 60.0),
             ccrs.PlateCarree()),
        Area('troppac', 'TROPICAL PACIFIC', (-165.0, -80.0, -25.0, 45.0),
             ccrs.PlateCarree()),
        Area('gulf', 'GULF OF MEXICO', (-105.0, -70.0, 10.0, 40.0),
             ccrs.PlateCarree()),
        Area('carib', 'CARIBBEAN SEA', (-100.0, -50.0, 0.0, 40.0),
             ccrs.PlateCarree()),
        Area('sthepac', 'SOUTH E PACIFIC', (-170.0, -70.0, -60.0, 0.0),
             ccrs.PlateCarree()),
        Area('natlmed', 'N ATL/MEDITER', (-30.0, 70.0, 0.0, 65.0),
             ccrs.PlateCarree()),
        Area('atlhur', 'ATLANTIC HURR', (-96.0, -6.0, 4.0, 3.0),
             ccrs.NorthPolarStereo(central_longitude=-90)),
        Area('nam', 'N AMERICA', (-144, -50, 12, 75),
             ccrs.NorthPolarStereo(central_longitude=-105)),
        Area('sam', 'S AMERICA', (-120.0, -20.0, -60.0, 20.0),
             ccrs.PlateCarree()),
        Area('samps', 'S AMERICA (PS)', (-100.0, -34.0, -58.0, 15.0),
             ccrs.RotatedPole(pole_latitude=90, pole_longitude=0,
                              central_rotated_longitude=-70)),
        Area('eur', 'EUROPE', (-16.0, 80.0, 24.0, 52.0),
             ccrs.NorthPolarStereo(central_longitude=15)),
        Area('natl', 'N ATLANTIC', (-110.0, 20.1, 15.0, 70.0),
             ccrs.PlateCarree()),
        Area('watl', 'W ATLANTIC', (-84.0, -38.0, 25.0, 46.0),
             ccrs.NorthPolarStereo(central_longitude=-67)),
        Area('tatl', 'TROPICAL ATLANTIC', (-90.0, -15.0, -10.0, 35.0),
             ccrs.PlateCarree()),
        Area('npac', 'N PACIFIC', (102.0, 250.0, -12.0, 60.0),
             ccrs.PlateCarree(central_longitude=-180)),
        Area('spac', 'S PACIFIC', (102.0, 290.0, -60.0, 20.0),
             ccrs.PlateCarree(central_longitude=-180)),
        Area('tpac', 'TROPICAL PACIFIC', (-165.0, -75.0, -10.0, 40.0),
             ccrs.PlateCarree()),
        Area('epac', 'E PACIFIC', (-150.0, -98.0, 12.0, 60.0),
             ccrs.RotatedPole(pole_latitude=90, pole_longitude=0,
                              central_rotated_longitude=-130)),
        Area('wpac', 'W PACIFIC', (130.0, 240.0, 0.0, 63.0),
             ccrs.PlateCarree(central_longitude=-180)),
        Area('mpac', 'MARINE PACIFIC', (128.0, 252.0, 15.0, 71.95),
             ccrs.PlateCarree(central_longitude=-180)),
        Area('americas', 'ICAO MERC AREA A', (-137.4, -12.6, -54.0, 67.0),
             ccrs.PlateCarree()),
        Area('atlantic', 'ICAO MERC AREA B1', (-125.0, 40.0, -45.5, 62.7),
             ccrs.PlateCarree()),
        Area('africa', 'ICAO MERC AREA C', (-35.0, 70.0, -45.0, 75.0),
             ccrs.PlateCarree()),
        Area('indocean', 'ICAO MERC AREA D', (-15.0, 132.0, -27.0, 63.0),
             ccrs.PlateCarree()),
        Area('oceania', 'ICAO MERC AREA E', (25.0, 180.0, -54.0, 40.0),
             ccrs.PlateCarree()),
        Area('trop_pac', 'ICAO MERC AREA F', (100.0, 250, -52.7, 50.0),
             ccrs.PlateCarree()),
        Area('asia_ps', 'ICAO P.S. AREA G', (34.8, 157.2, -0.8, 13.7),
             ccrs.NorthPolarStereo(central_longitude=65)),
        Area('na_ps', 'ICAO P.S. AREA H', (-79.1, 56.7, 1.6, 25.2),
             ccrs.NorthPolarStereo(central_longitude=-45)),
        Area('npac_ps', 'ICAO P.S. AREA I', (166.24, -60.62, -6.74, 33.32),
             ccrs.NorthPolarStereo(central_longitude=-155)),
        Area('antarctica', 'ICAO P.S. AREA J', (106.8, -101.1, -27.6, 0.8),
             ccrs.SouthPolarStereo(central_longitude=-142)),
        Area('europe', 'ICAO P.S. AREA EU', (-21.6, 68.4, 21.4, 58.7),
             ccrs.Stereographic(central_latitude=90, central_longitude=11)),
        Area('middle_east', 'ICAO MERC AREA ME', (17.0, 70.0, 10.0, 44.0),
             ccrs.PlateCarree()),
        Area('india', 'ICAO MERC AREA AS', (53.0, 108.0, 0.0, 36.0),
             ccrs.PlateCarree()),
        Area('ag', 'ARGENTINA', (-80.0, -53.0, -56.0, -20.0),
             ccrs.SouthPolarStereo(central_longitude=-67)),
        Area('ah', 'AFGHANISTAN', (60.0, 77.0, 27.0, 40.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=69)),
        Area('afrca', 'AFRICA', (-25.0, 59.4, -36.0, 41.0),
             ccrs.PlateCarree()),
        Area('ai', 'ASCENSION ISLAND', (-14.3, -14.1, -8.0, -7.8),
             ccrs.PlateCarree()),
        Area('alba', 'ALBANIA', (18.0, 23.0, 39.0, 43.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=21)),
        Area('alge', 'ALGERIA', (-9.0, 12.0, 15.0, 38.0),
             ccrs.PlateCarree()),
        Area('an', 'ANGOLA', (10.0, 25.0, -20.0, -5.0),
             ccrs.PlateCarree()),
        Area('antl', 'LESSER ANTILLES', (-70.0, -58.0, 11.0, 19.0),
             ccrs.PlateCarree()),
        Area('antg', 'GREATER ANTILLES', (-86.0, -65.0, 17.0, 25.0),
             ccrs.PlateCarree()),
        Area('atg', 'ANTIGUA & BARBUDA', (-62.0, -61.6, 16.9, 17.75),
             ccrs.PlateCarree()),
        Area('au', 'AUSTRALIA', (110.0, 160.0, -40.0, -6.5),
             ccrs.SouthPolarStereo(central_longitude=140)),
        Area('azor', 'AZORES', (-27.6, -23.0, 36.0, 41.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-25)),
        Area('ba', 'BAHAMAS', (-80.5, -72.5, 22.5, 28.5),
             ccrs.PlateCarree()),
        Area('be', 'BERMUDA', (-64.9, -64.5, 32.2, 32.6),
             ccrs.PlateCarree()),
        Area('bel', 'BELGIUM', (2.5, 6.5, 49.4, 51.6),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=5)),
        Area('bf', 'BRUNEI', (113.0, 116.0, 4.0, 5.5),
             ccrs.PlateCarree()),
        Area('bfa', 'BURKINA FASO', (-6.0, 3.0, 9.0, 15.1),
             ccrs.PlateCarree()),
        Area('bh', 'BELIZE', (-89.3, -88.1, 15.7, 18.5),
             ccrs.PlateCarree()),
        Area('bi', 'BURUNDI', (29.0, 30.9, -4.6, -2.2),
             ccrs.PlateCarree()),
        Area('bj', 'BENIN', (0.0, 5.0, 6.0, 12.6),
             ccrs.PlateCarree()),
        Area('bn', 'BAHRAIN', (50.0, 51.0, 25.5, 27.1),
             ccrs.PlateCarree()),
        Area('bo', 'BOLIVIA', (-72.0, -50.0, -24.0, -8.0),
             ccrs.PlateCarree()),
        Area('bots', 'BOTSWANA', (19.0, 29.6, -27.0, -17.0),
             ccrs.PlateCarree()),
        Area('br', 'BARBADOS', (-62.5, -56.5, 12.45, 13.85),
             ccrs.PlateCarree()),
        Area('bt', 'BRITISH INDIAN OC', (71.25, 72.6, -7.5, -5.0),
             ccrs.PlateCarree()),
        Area('bu', 'BULGARIA', (22.0, 30.0, 40.0, 45.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=26)),
        Area('bv', 'BOUVET ISLANDS', (3.0, 4.0, -55.0, -54.0),
             ccrs.SouthPolarStereo(central_longitude=4)),
        Area('bw', 'BANGLADESH', (87.0, 93.0, 20.8, 27.0),
             ccrs.PlateCarree()),
        Area('by', 'BYELORUSSIA', (19.0, 33.0, 51.0, 60.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=26)),
        Area('bz', 'BRAZIL', (-75.0, -30.0, -35.0, 5.0),
             ccrs.PlateCarree()),
        Area('cais', 'CANTON ISLAND', (-172.0, -171.0, -3.0, -2.0),
             ccrs.PlateCarree()),
        Area('nwcar', 'NWX CARIBBEAN', (-120.0, -50.0, -15.0, 35.0),
             ccrs.PlateCarree()),
        Area('cari', 'CARIBBEAN', (-103.0, -53.0, 3.0, 36.0),
             ccrs.PlateCarree()),
        Area('cb', 'CHAD', (13.0, 25.0, 7.0, 24.0),
             ccrs.PlateCarree()),
        Area('ce', 'CENTRAL AFRICA', (14.0, 29.0, 2.0, 11.5),
             ccrs.PlateCarree()),
        Area('cg', 'CONGO', (10.0, 20.0, -6.0, 5.0),
             ccrs.PlateCarree()),
        Area('ch', 'CHILE', (-80.0, -66.0, -56.0, -15.0),
             ccrs.SouthPolarStereo(central_longitude=-73)),
        Area('ci', 'CHINA', (85.0, 145.0, 14.0, 48.5),
             ccrs.NorthPolarStereo(central_longitude=110)),
        Area('cm', 'CAMEROON', (7.5, 17.1, 1.0, 14.0),
             ccrs.PlateCarree()),
        Area('colm', 'COLUMBIA', (-81.0, -65.0, -5.0, 14.0),
             ccrs.PlateCarree()),
        Area('cr', 'CANARY ISLANDS', (-19.0, -13.0, 27.0, 30.0),
             ccrs.PlateCarree()),
        Area('cs', 'COSTA RICA', (-86.5, -81.5, 8.2, 11.6),
             ccrs.PlateCarree()),
        Area('cu', 'CUBA', (-85.0, -74.0, 19.0, 24.0),
             ccrs.PlateCarree()),
        Area('cv', 'CAPE VERDE ISLAND', (-26.0, -22.0, 14.0, 18.0),
             ccrs.PlateCarree()),
        Area('cy', 'CYPRUS', (32.0, 35.0, 34.0, 36.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=34)),
        Area('cz', 'CZECH REPUBLIC', (8.9, 22.9, 47.4, 52.4),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=16)),
        Area('dj', 'DJIBOUTI', (41.5, 44.1, 10.5, 13.1),
             ccrs.PlateCarree()),
        Area('dl', 'GERMANY', (4.8, 16.8, 47.0, 55.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=11)),
        Area('dn', 'DENMARK', (8.0, 11.0, 54.0, 58.6),
             ccrs.NorthPolarStereo(central_longitude=10)),
        Area('do', 'DOMINICA', (-61.6, -61.2, 15.2, 15.8),
             ccrs.PlateCarree()),
        Area('dr', 'DOMINICAN REPUBLIC', (-72.2, -68.0, 17.5, 20.2),
             ccrs.PlateCarree()),
        Area('eg', 'EGYPT', (24.0, 37.0, 21.0, 33.0),
             ccrs.PlateCarree()),
        Area('eq', 'ECUADOR', (-85.0, -74.0, -7.0, 3.0),
             ccrs.PlateCarree()),
        Area('er', 'UAE', (50.0, 57.0, 22.0, 26.6),
             ccrs.PlateCarree()),
        Area('es', 'EL SALVADOR', (-90.3, -87.5, 13.0, 14.6),
             ccrs.PlateCarree()),
        Area('et', 'ETHIOPIA', (33.0, 49.0, 2.0, 19.0),
             ccrs.PlateCarree()),
        Area('fa', 'FAEROES', (-8.0, -6.0, 61.0, 63.0),
             ccrs.NorthPolarStereo(central_longitude=-7)),
        Area('fg', 'FRENCH GUIANA', (-55.0, -49.0, 1.0, 7.0),
             ccrs.PlateCarree()),
        Area('fi', 'FINLAND', (20.9, 35.1, 59.0, 70.6),
             ccrs.NorthPolarStereo(central_longitude=28)),
        Area('fj', 'FIJI ISLANDS', (176.0, 181.0, 16.0, 19.0),
             ccrs.PlateCarree(central_longitude=-180)),
        Area('fk', 'FALKLAND ISLANDS', (-61.3, -57.5, -53.0, -51.0),
             ccrs.SouthPolarStereo(central_longitude=-57)),
        Area('fn', 'NIGER', (0.0, 17.0, 11.0, 24.0),
             ccrs.PlateCarree()),
        Area('fr', 'FRANCE', (-5.0, 11.0, 41.0, 51.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=3)),
        Area('gb', 'GAMBIA', (-17.1, -13.5, 13.0, 14.6),
             ccrs.PlateCarree()),
        Area('gc', 'CAYMAN ISLANDS', (-82.8, -77.6, 17.9, 21.1),
             ccrs.PlateCarree()),
        Area('gh', 'GHANA', (-4.5, 1.5, 4.0, 12.0),
             ccrs.PlateCarree()),
        Area('gi', 'GIBRALTAR', (-8.0, -4.0, 35.0, 38.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-6)),
        Area('gl', 'GREENLAND', (-60, -20, 58.3, 85),
             ccrs.NorthPolarStereo(central_longitude=-45)),
        Area('glp', 'GUADALOUPE', (-64.2, -59.8, 14.8, 19.2),
             ccrs.PlateCarree()),
        Area('gm', 'GUAM', (144.5, 145.1, 13.0, 14.0),
             ccrs.PlateCarree()),
        Area('gn', 'GUINEA', (2.0, 16.0, 3.5, 15.5),
             ccrs.PlateCarree()),
        Area('go', 'GABON', (8.0, 14.5, -4.6, 3.0),
             ccrs.PlateCarree()),
        Area('gr', 'GREECE', (20.0, 27.6, 34.0, 42.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=24)),
        Area('gu', 'GUATEMALA', (-95.6, -85.0, 10.5, 21.1),
             ccrs.PlateCarree()),
        Area('gw', 'GUINEA-BISSAU', (-17.5, -13.5, 10.8, 12.8),
             ccrs.PlateCarree()),
        Area('gy', 'GUYANA', (-62.0, -55.0, 0.0, 10.0),
             ccrs.PlateCarree()),
        Area('ha', 'HAITI', (-75.0, -71.0, 18.0, 20.0),
             ccrs.PlateCarree()),
        Area('he', 'ST HELENA', (-6.1, -5.5, -16.3, -15.5),
             ccrs.PlateCarree()),
        Area('hk', 'HONG KONG', (113.5, 114.7, 22.0, 23.0),
             ccrs.PlateCarree()),
        Area('ho', 'HONDURAS', (-90.0, -83.0, 13.0, 16.6),
             ccrs.PlateCarree()),
        Area('hu', 'HUNGARY', (16.0, 23.0, 45.5, 49.1),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=20)),
        Area('ic', 'COMOROS', (43.0, 45.0, -13.2, -11.0),
             ccrs.PlateCarree()),
        Area('icel', 'ICELAND', (-24.1, -11.5, 63.0, 67.5),
             ccrs.NorthPolarStereo(central_longitude=-18)),
        Area('ie', 'IRELAND', (-11.1, -4.5, 50.0, 55.6),
             ccrs.NorthPolarStereo(central_longitude=-8)),
        Area('inda', 'INDIA', (67.0, 92.0, 4.2, 36.0),
             ccrs.PlateCarree()),
        Area('indo', 'INDONESIA', (95.0, 141.0, -8.0, 6.0),
             ccrs.PlateCarree()),
        Area('iq', 'IRAQ', (38.0, 50.0, 29.0, 38.0),
             ccrs.PlateCarree()),
        Area('ir', 'IRAN', (44.0, 65.0, 25.0, 40.0),
             ccrs.PlateCarree()),
        Area('is', 'ISRAEL', (34.0, 37.0, 29.0, 34.0),
             ccrs.PlateCarree()),
        Area('iv', 'IVORY COAST', (-9.0, -2.0, 4.0, 11.0),
             ccrs.PlateCarree()),
        Area('iw', 'WEST BANK', (34.8, 35.6, 31.2, 32.6),
             ccrs.PlateCarree()),
        Area('iy', 'ITALY', (6.6, 20.6, 35.6, 47.2),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=14)),
        Area('jd', 'JORDAN', (34.0, 39.6, 29.0, 33.6),
             ccrs.PlateCarree()),
        Area('jm', 'JAMAICA', (-80.0, -76.0, 16.0, 19.0),
             ccrs.PlateCarree()),
        Area('jp', 'JAPAN', (123.0, 155.0, 24.0, 47.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=139)),
        Area('ka', 'CAROLINE ISLANDS', (131.0, 155.0, 1.0, 9.6),
             ccrs.PlateCarree()),
        Area('kash', 'JAMMU & KASHMIR', (74.0, 78.0, 32.0, 35.0),
             ccrs.PlateCarree()),
        Area('kb', 'KIRIBATI', (172.0, 177.0, -3.0, 3.2),
             ccrs.PlateCarree()),
        Area('khm', 'CAMBODIA', (102.0, 108.0, 10.0, 15.0),
             ccrs.PlateCarree()),
        Area('ki', 'CHRISTMAS ISLAND', (105.2, 106.2, -11.0, -10.0),
             ccrs.PlateCarree()),
        Area('kn', 'KENYA', (32.5, 42.1, -6.0, 6.0),
             ccrs.PlateCarree()),
        Area('kna', 'ST KITTS & NEVIS', (-62.9, -62.4, 17.0, 17.5),
             ccrs.PlateCarree()),
        Area('ko', 'KOREA', (124.0, 131.5, 33.0, 43.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=128)),
        Area('ku', 'COOK ISLANDS', (-168.0, -155.0, -24.1, -6.1),
             ccrs.PlateCarree()),
        Area('kw', 'KUWAIT', (46.5, 48.5, 28.5, 30.5),
             ccrs.PlateCarree()),
        Area('laos', 'LAOS', (100.0, 108.0, 13.5, 23.1),
             ccrs.PlateCarree()),
        Area('lb', 'LEBANON', (34.5, 37.1, 33.0, 35.0),
             ccrs.PlateCarree()),
        Area('lc', 'ST LUCIA', (60.9, 61.3, 13.25, 14.45),
             ccrs.PlateCarree()),
        Area('li', 'LIBERIA', (-12.0, -7.0, 4.0, 9.0),
             ccrs.PlateCarree()),
        Area('ln', 'LINE ISLANDS', (-162.1, -154.9, -4.2, 6.0),
             ccrs.PlateCarree()),
        Area('ls', 'LESOTHO', (27.0, 29.6, -30.6, -28.0),
             ccrs.PlateCarree()),
        Area('lt', 'LIECHTENSTEIN', (9.3, 9.9, 47.0, 47.6),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=10)),
        Area('lux', 'LUXEMBOURG', (5.6, 6.6, 49.35, 50.25),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=6)),
        Area('ly', 'LIBYA', (8.0, 26.0, 19.0, 35.0),
             ccrs.PlateCarree()),
        Area('maar', 'ST MAARTEN', (-63.9, -62.3, 17.0, 18.6),
             ccrs.PlateCarree()),
        Area('made', 'MADEIRA', (-17.3, -16.5, 32.6, 33.0),
             ccrs.PlateCarree()),
        Area('mala', 'MALAYSIA', (100.0, 119.6, 1.0, 8.0),
             ccrs.PlateCarree()),
        Area('mali', 'MALI', (-12.5, 6.0, 8.5, 25.5),
             ccrs.PlateCarree()),
        Area('maur', 'MAURITIUS', (57.2, 57.8, -20.7, -19.9),
             ccrs.PlateCarree()),
        Area('maut', 'MAURITANIA', (-17.1, -4.5, 14.5, 28.1),
             ccrs.PlateCarree()),
        Area('mc', 'MOROCCO', (-13.0, -1.0, 25.0, 36.0),
             ccrs.PlateCarree()),
        Area('mg', 'MADAGASCAR', (43.0, 50.6, -25.6, -12.0),
             ccrs.PlateCarree()),
        Area('mh', 'MARSHALL ISLANDS', (160.0, 172.0, 4.5, 12.1),
             ccrs.PlateCarree()),
        Area('ml', 'MALTA', (14.3, 14.7, 35.8, 36.0),
             ccrs.PlateCarree()),
        Area('mmr', 'MYANMAR', (92.0, 102.0, 7.5, 28.5),
             ccrs.PlateCarree()),
        Area('mong', 'MONGOLIA', (87.5, 123.1, 38.5, 52.6),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=105)),
        Area('mr', 'MARTINIQUE', (-61.2, -60.8, 14.3, 15.1),
             ccrs.PlateCarree()),
        Area('mu', 'MACAO', (113.0, 114.0, 22.0, 23.0),
             ccrs.PlateCarree()),
        Area('mv', 'MALDIVE ISLANDS', (70.1, 76.1, -6.0, 10.0),
             ccrs.PlateCarree()),
        Area('mw', 'MALAWI', (32.5, 36.1, -17.0, -9.0),
             ccrs.PlateCarree()),
        Area('mx', 'MEXICO', (-119.0, -83.0, 13.0, 34.0),
             ccrs.PlateCarree()),
        Area('my', 'MARIANNA ISLANDS', (142.5, 148.5, 9.0, 25.0),
             ccrs.PlateCarree()),
        Area('mz', 'MOZAMBIQUE', (29.0, 41.0, -26.5, -9.5),
             ccrs.PlateCarree()),
        Area('nama', 'NAMIBIA', (11.0, 25.0, -29.5, -16.5),
             ccrs.PlateCarree()),
        Area('ncal', 'NEW CALEDONIA', (158.0, 172.0, -23.0, -18.0),
             ccrs.PlateCarree()),
        Area('ng', 'NEW GUINEA', (130.0, 152.0, -11.0, 0.0),
             ccrs.PlateCarree()),
        Area('ni', 'NIGERIA', (2.0, 14.6, 3.0, 14.0),
             ccrs.PlateCarree()),
        Area('nk', 'NICARAGUA', (-88.0, -83.0, 10.5, 15.1),
             ccrs.PlateCarree()),
        Area('nl', 'NETHERLANDS', (3.5, 7.5, 50.5, 54.1),
             ccrs.NorthPolarStereo(central_longitude=6)),
        Area('no', 'NORWAY', (3.0, 35.0, 57.0, 71.5),
             ccrs.NorthPolarStereo(central_longitude=19)),
        Area('np', 'NEPAL', (80.0, 89.0, 25.0, 31.0),
             ccrs.PlateCarree()),
        Area('nw', 'NAURU', (166.4, 167.4, -1.0, 0.0),
             ccrs.PlateCarree()),
        Area('nz', 'NEW ZEALAND', (165.0, 179.0, -48.0, -33.0),
             ccrs.SouthPolarStereo(central_longitude=172)),
        Area('om', 'OMAN', (52.0, 60.0, 16.0, 25.6),
             ccrs.PlateCarree()),
        Area('os', 'AUSTRIA', (9.0, 18.0, 46.0, 50.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=14)),
        Area('pf', 'FRENCH POLYNESIA', (-154.0, -134.0, -28.0, -8.0),
             ccrs.PlateCarree()),
        Area('ph', 'PHILIPINES', (116.0, 127.0, 4.0, 21.0),
             ccrs.PlateCarree()),
        Area('pi', 'PHOENIX ISLANDS', (-177.5, -167.5, -9.0, 1.0),
             ccrs.PlateCarree()),
        Area('pk', 'PAKISTAN', (60.0, 78.0, 23.0, 37.0),
             ccrs.PlateCarree()),
        Area('pl', 'POLAND', (14.0, 25.0, 48.5, 55.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=20)),
        Area('pm', 'PANAMA', (-83.0, -77.0, 7.0, 10.0),
             ccrs.PlateCarree()),
        Area('po', 'PORTUGAL', (-10.0, -4.0, 36.5, 42.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-7)),
        Area('pr', 'PERU', (-82.0, -68.0, -20.0, 5.0),
             ccrs.PlateCarree()),
        Area('pt', 'PITCAIRN ISLANDS', (-130.6, -129.6, -25.56, -24.56),
             ccrs.PlateCarree()),
        Area('pu', 'PUERTO RICO', (-67.5, -65.5, 17.5, 18.5),
             ccrs.PlateCarree()),
        Area('py', 'PARAGUAY', (-65.0, -54.0, -32.0, -17.0),
             ccrs.PlateCarree()),
        Area('qg', 'EQUATORIAL GUINEA', (7.0, 12.0, -2.0, 3.0),
             ccrs.PlateCarree()),
        Area('qt', 'QATAR', (50.0, 52.0, 24.0, 27.0),
             ccrs.PlateCarree()),
        Area('ra', 'RUSSIA', (40.0, 190.0, 35.0, 70.0),
             ccrs.LambertConformal(standard_parallels=[40, 70], central_longitude=105)),
        Area('re', 'REUNION', (55.0, 56.0, -21.5, -20.5),
             ccrs.PlateCarree()),
        Area('riro', 'RIO DE ORO', (-18.0, -12.0, 17.5, 27.5),
             ccrs.PlateCarree()),
        Area('ro', 'ROMANIA', (19.0, 31.0, 42.5, 48.5),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=25)),
        Area('rw', 'RWANDA', (29.0, 31.0, -3.0, -1.0),
             ccrs.PlateCarree()),
        Area('saudi', 'SAUDI ARABIA', (4.6, 92.5, -13.2, 60.3),
             ccrs.PlateCarree()),
        Area('sb', 'SRI LANKA', (79.0, 83.0, 5.0, 10.0),
             ccrs.PlateCarree()),
        Area('seyc', 'SEYCHELLES ISLANDS', (55.0, 56.0, -5.0, -4.0),
             ccrs.PlateCarree()),
        Area('sg', 'SENEGAL', (-18.0, -10.0, 12.0, 17.0),
             ccrs.PlateCarree()),
        Area('si', 'SOMALIA', (39.5, 52.1, -4.5, 13.5),
             ccrs.PlateCarree()),
        Area('sk', 'SARAWAK', (109.5, 119.3, 1.0, 7.0),
             ccrs.PlateCarree()),
        Area('sl', 'SIERRA LEONE', (-13.6, -10.2, 6.9, 10.1),
             ccrs.PlateCarree()),
        Area('sm', 'SURINAME', (-59.0, -53.0, 1.0, 6.0),
             ccrs.PlateCarree()),
        Area('sn', 'SWEDEN', (10.0, 25.0, 55.0, 69.6),
             ccrs.NorthPolarStereo(central_longitude=18)),
        Area('so', 'SOLOMON ISLANDS', (156.0, 167.0, -12.0, -6.0),
             ccrs.PlateCarree()),
        Area('sp', 'SPAIN', (-10.0, 6.0, 35.0, 44.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-2)),
        Area('sr', 'SINGAPORE', (103.0, 105.0, 1.0, 2.0),
             ccrs.PlateCarree()),
        Area('su', 'SUDAN', (21.5, 38.5, 3.5, 23.5),
             ccrs.PlateCarree()),
        Area('sv', 'SWAZILAND', (30.5, 33.1, -27.5, -25.3),
             ccrs.PlateCarree()),
        Area('sw', 'SWITZERLAND', (5.9, 10.5, 45.8, 48.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=8)),
        Area('sy', 'SYRIA', (35.0, 42.6, 32.0, 37.6),
             ccrs.PlateCarree()),
        Area('tanz', 'TANZANIA', (29.0, 40.6, -13.0, 0.0),
             ccrs.PlateCarree()),
        Area('td', 'TRINIDAD & TOBAGO', (-62.1, -60.5, 10.0, 11.6),
             ccrs.PlateCarree()),
        Area('tg', 'TOGO', (-0.5, 2.5, 5.0, 12.0),
             ccrs.PlateCarree()),
        Area('th', 'THAILAND', (97.0, 106.0, 5.0, 21.0),
             ccrs.PlateCarree()),
        Area('ti', 'TURKS ISLANDS', (-71.6, -70.6, 21.0, 22.0),
             ccrs.PlateCarree()),
        Area('tk', 'TOKELAU ISLANDS', (-173.0, -171.0, -11.5, -7.5),
             ccrs.PlateCarree()),
        Area('to', 'TONGA', (-178.5, -170.5, -22.0, -15.0),
             ccrs.PlateCarree()),
        Area('tp', 'SAO TOME-PRINCIPE', (6.0, 7.6, 0.0, 2.0),
             ccrs.PlateCarree()),
        Area('ts', 'TUNISIA', (7.0, 13.0, 30.0, 38.0),
             ccrs.PlateCarree()),
        Area('tu', 'TURKEY', (25.0, 48.0, 34.1, 42.1),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=37)),
        Area('tv', 'TUVALU', (176.0, 180.0, -11.0, -5.0),
             ccrs.PlateCarree()),
        Area('tw', 'TAIWAN', (120.0, 122.0, 21.9, 25.3),
             ccrs.PlateCarree()),
        Area('ug', 'UGANDA', (29.0, 35.0, -3.5, 5.5),
             ccrs.PlateCarree()),
        Area('uk', 'UNITED KINGDOM', (-11.0, 5.0, 49.0, 60.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=-3)),
        Area('ur', 'UKRAINE', (20.0, 41.0, 45.0, 52.0),
             ccrs.NorthPolarStereo(central_longitude=20)),
        Area('uy', 'URUGUAY', (-60.0, -52.0, -35.5, -29.5),
             ccrs.SouthPolarStereo(central_longitude=-56)),
        Area('vanu', 'VANUATU', (167.0, 170.0, -21.0, -13.0),
             ccrs.PlateCarree()),
        Area('vi', 'VIRGIN ISLANDS', (-65.5, -64.0, 16.6, 19.6),
             ccrs.PlateCarree()),
        Area('vk', 'SLOVAKIA', (13.8, 25.8, 46.75, 50.75),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=20)),
        Area('vn', 'VENEZUELA', (-75.0, -60.0, -2.0, 14.0),
             ccrs.PlateCarree()),
        Area('vs', 'VIETNAM', (102.0, 110.0, 8.0, 24.0),
             ccrs.PlateCarree()),
        Area('wk', 'WAKE ISLAND', (166.1, 167.1, 18.8, 19.8),
             ccrs.PlateCarree()),
        Area('ye', 'YEMEN', (42.5, 54.1, 12.5, 19.1),
             ccrs.PlateCarree()),
        Area('yg', 'YUGOSLAVIA', (13.5, 24.6, 40.0, 47.0),
             ccrs.LambertConformal(standard_parallels=[30, 60], central_longitude=19)),
        Area('za', 'SOUTH AFRICA', (16.0, 34.0, -36.0, -22.0),
             ccrs.PlateCarree()),
        Area('zb', 'ZAMBIA', (21.0, 35.0, -20.0, -7.0),
             ccrs.PlateCarree()),
        Area('zm', 'WESTERN SAMOA', (170.5, 173.5, -15.0, -13.0),
             ccrs.PlateCarree()),
        Area('zr', 'ZAIRE', (12.0, 31.6, -14.0, 6.0),
             ccrs.PlateCarree()),
        Area('zw', 'ZIMBABWE', (25.0, 34.0, -22.9, -15.5),
             ccrs.PlateCarree())
    ]

    named_areas = {data.name: data for data in area_data}
except ImportError:
    # If no Cartopy is present, we just don't have plot areas
    pass

__all__ = ['named_areas']
